% Comparison Of Modelling Functions Example
%
% This example provides a short comparison between the simulation functions
% kspaceFirstOrder2D and kspaceSecondOrder. It builds on the Homogeneous
% Propagation Medium and Using A Binary Sensor Mask examples. 
%
% author: Bradley Treeby
% date: 27th October 2010
% last update: 29th October 2010
%  
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010, 2011 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>. 

clear all;

example_number = 1;
% 1: non-absorbing medium, no absorbing boundary layer
% 2: non-absorbing medium, using PML and ExpandGrid
% 3: absorbing medium, no absorbing boundary layer
% 4: absorbing medium, using PML and ExpandGrid

% =========================================================================
% SIMULATION
% =========================================================================

% create the computational grid
Nx = 128;           % number of pixels in the x (column) direction
Nz = 128;           % number of pixels in the z (row) direction
dx = 0.1e-3;        % pixel width [m]
dz = 0.1e-3;        % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% define the properties of the propagation medium
medium.sound_speed = 1500;  % [m/s]
if example_number > 2
    medium.alpha_power = 1.5;   % [dB/(MHz^y cm)]
    medium.alpha_coeff = 0.75;  % [dB/(MHz^y cm)]
end

% create the time array
t_end = 6e-6;
kgrid.t_array = makeTime(kgrid, medium.sound_speed, 0.3, t_end);

% create initial pressure distribution using makeDisc
disc_magnitude = 5; % [au]
disc_x_pos = 50;    % [pixels]
disc_z_pos = 50;    % [pixels]
disc_radius = 8;    % [pixels]
disc_1 = disc_magnitude*makeDisc(Nx, Nz, disc_x_pos, disc_z_pos, disc_radius);

disc_magnitude = 3; % [au]
disc_x_pos = 60;    % [pixels]
disc_z_pos = 80;    % [pixels]
disc_radius = 5;    % [pixels]
disc_2 = disc_magnitude*makeDisc(Nx, Nz, disc_x_pos, disc_z_pos, disc_radius);

source.p0 = disc_1 + disc_2;

% define a centered circular sensor pushed right to the edge of the grid
sensor_radius = 6.3e-3;   % [m]
num_sensor_points = 50;
sensor.mask = makeCartCircle(sensor_radius, num_sensor_points);

% convert the cartesian sensor mask to a binary sensor mask
sensor.mask = cart2grid(kgrid, sensor.mask);

if example_number == 1  || example_number == 3      % no absorbing boundary layer

    % run the simulation using the first order code
    sensor_data_first_order = kspaceFirstOrder2D(kgrid, medium, source, sensor, 'PMLAlpha', 0);

    % run the simulation using the second order code
    sensor_data_second_order = kspaceSecondOrder(kgrid, medium, source, sensor, 'ExpandGrid', false);
    
elseif example_number == 2  || example_number == 4  % using PML and ExpandGrid
    
    % run the simulation using the first order code
    sensor_data_first_order = kspaceFirstOrder2D(kgrid, medium, source, sensor, 'PMLInside', false);

    % run the simulation using the second order code
    sensor_data_second_order = kspaceSecondOrder(kgrid, medium, source, sensor, 'ExpandGrid', true);
end

% =========================================================================
% VISUALISATION
% =========================================================================

% plot a single time series
figure;
[t_sc, t_scale, t_prefix] = scaleSI(kgrid.t_array(end));
subplot(2, 1, 1), plot(kgrid.t_array*t_scale, sensor_data_second_order(1, :), 'k-', kgrid.t_array*t_scale, sensor_data_first_order(1, :), 'bx');
set(gca, 'YLim', [-1, 1.5]);
xlabel(['time [' t_prefix 's]']);
ylabel('pressure [au]');
legend('kspaceSecondOrder', 'kspaceFirstOrder2D', 'Location', 'NorthWest');
title('Recorded Signals');

subplot(2, 1, 2), plot(kgrid.t_array*t_scale, sensor_data_second_order(1, :) - sensor_data_first_order(1, :), 'k-');
xlabel(['time [' t_prefix 's]']);
ylabel('pressure [au]');
title('Difference In Recorded Signals');